import 'dart:async';

import 'package:rad/rad.dart';

import 'package:photogram_admin_cp/import/bloc.dart';
import 'package:photogram_admin_cp/import/core.dart';
import 'package:photogram_admin_cp/import/data.dart';

/*
|--------------------------------------------------------------------------
| bloc:
|--------------------------------------------------------------------------
*/

class AuthBloc {
  static AuthBloc of(BuildContext context) => AppProvider.of(context).auth;

  /*
  |--------------------------------------------------------------------------
  | stream controllers:
  |--------------------------------------------------------------------------
  */

  final _stateController = StreamController<AuthState>.broadcast();

  final _eventController = StreamController<AuthEvent>();

  /*
  |--------------------------------------------------------------------------
  | exposed stream
  |--------------------------------------------------------------------------
  */

  Stream<AuthState> get stream => _stateController.stream;

  /*
  |--------------------------------------------------------------------------
  | exposed authed admin
  |--------------------------------------------------------------------------
  */

  AdminModel get getCurrentAdmin => _authedAdmin;

  /*
  |--------------------------------------------------------------------------
  | hold authed admin here
  |--------------------------------------------------------------------------
  */

  var _authedAdmin = AdminModel();

  /*
  |--------------------------------------------------------------------------
  | constructor:
  |--------------------------------------------------------------------------
  */

  AuthBloc() {
    AppLogger.blockInfo("AuthBloc: Init");
    _eventController.stream.listen(_handleEvent);
  }

  /*
  |--------------------------------------------------------------------------
  | push events [exposed]:
  |--------------------------------------------------------------------------
  */

  void pushEvent(AuthEvent event) {
    AppLogger.blockInfo("AuthBloc [UI]: Event ~> ${event.runtimeType}");
    _eventController.sink.add(event);
  }

  /*
  |--------------------------------------------------------------------------
  | push events [internal]:
  |--------------------------------------------------------------------------
  */

  void _pushEventInternally(AuthEvent event) {
    AppLogger.blockInfo("AuthBloc [Internally]: Event ~> ${event.runtimeType}");
    _eventController.sink.add(event);
  }

  /*
  |--------------------------------------------------------------------------
  | push state:
  |--------------------------------------------------------------------------
  */

  void _pushState(AuthState state) {
    AppLogger.blockInfo("AuthBloc: State ~> ${state.runtimeType}");

    // do internal state changes (if any)
    switch (state.runtimeType) {
      case AuthStateAuthed:
        _setAdminAsAuthed(state as AuthStateAuthed);
        break;

      case AuthStateLoggedOut:
        _setAdminAsLoggedOut(state as AuthStateLoggedOut);
        break;
    }

    _stateController.sink.add(state);
  }

  /*
  |--------------------------------------------------------------------------
  | handle events:
  |--------------------------------------------------------------------------
  */

  void _handleEvent(AuthEvent event) {
    AppLogger.blockInfo("AuthBloc: Handling event <~ ${event.runtimeType}");

    switch (event.runtimeType) {
      case AuthEventLogout:
        return _authEventLogout(event as AuthEventLogout);

      case AuthEventGetSession:
        return _authEventGetSession(event as AuthEventGetSession);

      case AuthEventPreparing:
        return _authEventPreparing(event as AuthEventPreparing);

      case AuthEventSetAuthedAdmin:
        return _authEventSetAuthedAdmin(event as AuthEventSetAuthedAdmin);
    }
  }

  /*
  |--------------------------------------------------------------------------
  | push preparing state
  |--------------------------------------------------------------------------
  */

  void _authEventPreparing(AuthEventPreparing event) async {
    _pushState(AuthStatePreparing(event.context, authedAdmin: event.authedAdmin));
  }

  /*
  |--------------------------------------------------------------------------
  | set logged in admin, admin is authed and received from some other component
  |--------------------------------------------------------------------------
  */

  void _authEventSetAuthedAdmin(AuthEventSetAuthedAdmin event) async {
    _pushState(AuthStateAuthed(event.context, authedAdmin: event.authedAdmin));
  }

  /*
  |--------------------------------------------------------------------------
  | try login from local repo(first visit usually):
  |--------------------------------------------------------------------------
  */

  void _authEventGetSession(AuthEventGetSession event) async {
    try {
      var responseModel = await AppProvider.of(event.context).apiRepo.authSession();

      if (responseModel.isNotResponse) {
        return _pushState(AuthStateNoNetwork(event.context));
      }

      if (responseModel.message != SUCCESS_MSG) throw Exception();

      if (!responseModel.contains(AdminTable.tableName)) throw Exception();

      var authedAdmin = AdminModel.fromJson(responseModel.first(AdminTable.tableName));

      if (!authedAdmin.isModel) throw Exception();

      _pushState(AuthStatePreparing(event.context, authedAdmin: authedAdmin));
    } catch (e) {
      // admin doesn't logged in or not exists in local repo
      // perform complete log out
      _pushEventInternally(AuthEventLogout(event.context));
    }
  }

  /*
  |--------------------------------------------------------------------------
  | logout event:
  |--------------------------------------------------------------------------
  */

  void _authEventLogout(AuthEventLogout event) {
    _pushState(AuthStateLoggedOut(event.context));
  }

  /*
  |--------------------------------------------------------------------------
  | helper functions:
  |--------------------------------------------------------------------------
  */

  void _setAdminAsAuthed(AuthStateAuthed state) {
    _authedAdmin = state.authedAdmin;
  }

  void _setAdminAsLoggedOut(AuthStateLoggedOut state) {
    _authedAdmin = AdminModel.none();
  }

  /*
  |--------------------------------------------------------------------------
  | dispose streams [exposed]:
  |--------------------------------------------------------------------------
  */

  void dispose() {
    _eventController.close();
    _stateController.close();

    AppLogger.blockInfo("AuthBloc: Dispose");
  }
}
